/*
 * Copyright (c) 2010, 2013, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */

package javafx.scene.effect;

import javafx.beans.property.DoubleProperty;
import javafx.beans.property.DoublePropertyBase;
import javafx.beans.property.ObjectProperty;
import javafx.scene.Node;

import com.sun.javafx.Utils;
import com.sun.javafx.effect.EffectDirtyBits;
import com.sun.javafx.geom.BaseBounds;
import com.sun.javafx.geom.transform.BaseTransform;
import com.sun.javafx.scene.BoundsAccessor;


/**
 * A filter that produces a sepia tone effect, similar to antique photographs.
 *
 * <p>
 * Example:
 * <pre><code>
 * SepiaTone sepiaTone = new SepiaTone();
 * sepiaTone.setLevel(0.7);
 *
 * Image image = new Image("boat.jpg");
 * ImageView imageView = new ImageView(image);
 * imageView.setFitWidth(200);
 * imageView.setPreserveRatio(true);
 * imageView.setEffect(sepiaTone);
 * </pre></code>
 * <p> The code above applied on this image: </p>
 * <p>
 * <img src="doc-files/photo.png"/>
 * </p>
 * <p> produces the following: </p>
 * <p>
 * <img src="doc-files/sepiatone.png"/>
 * </p>
 * @since JavaFX 2.0
 */
public class SepiaTone extends Effect {
    /**
     * Creates a new instance of SepiaTone with default parameters.
     */
    public SepiaTone() {}

    /**
     * Creates a new instance of SepiaTone with the specified level.
     * @param level the level value, which controls the intensity of the effect
     * @since JavaFX 2.1
     */
    public SepiaTone(double level) {
        setLevel(level);
    }

    @Override
    com.sun.scenario.effect.SepiaTone impl_createImpl() {
        return new com.sun.scenario.effect.SepiaTone();
    };
    /**
     * The input for this {@code Effect}.
     * If set to {@code null}, or left unspecified, a graphical image of
     * the {@code Node} to which the {@code Effect} is attached will be
     * used as the input.
     * @defaultValue null
     */
    private ObjectProperty<Effect> input;


    public final void setInput(Effect value) {
        inputProperty().set(value);
    }

    public final Effect getInput() {
        return input == null ? null : input.get();
    }

    public final ObjectProperty<Effect> inputProperty() {
        if (input == null) {
            input = new EffectInputProperty("input");
        }
        return input;
    }

    @Override
    boolean impl_checkChainContains(Effect e) {
        Effect localInput = getInput();
        if (localInput == null)
            return false;
        if (localInput == e)
            return true;
        return localInput.impl_checkChainContains(e);
    }

    /**
     * The level value, which controls the intensity of the sepia effect.
     * <pre>
     *       Min: 0.0f
     *       Max: 1.0f
     *   Default: 1.0f
     *  Identity: 0.0f
     * </pre>
     * @defaultValue 1.0f
     */
    private DoubleProperty level;


    public final void setLevel(double value) {
        levelProperty().set(value);
    }

    public final double getLevel() {
        return level == null ? 1 : level.get();
    }

    public final DoubleProperty levelProperty() {
        if (level == null) {
            level = new DoublePropertyBase(1) {

                @Override
                public void invalidated() {
                    markDirty(EffectDirtyBits.EFFECT_DIRTY);
                }

                @Override
                public Object getBean() {
                    return SepiaTone.this;
                }

                @Override
                public String getName() {
                    return "level";
                }
            };
        }
        return level;
    }

    @Override
    void impl_update() {
        Effect localInput = getInput();
        if (localInput != null) {
            localInput.impl_sync();
        }

        com.sun.scenario.effect.SepiaTone peer =
                (com.sun.scenario.effect.SepiaTone) impl_getImpl();
        peer.setInput(localInput == null ? null : localInput.impl_getImpl());
        peer.setLevel((float)Utils.clamp(0, getLevel(), 1));
    }

    /**
     * @treatAsPrivate implementation detail
     * @deprecated This is an internal API that is not intended for use and will be removed in the next version
     */
    @Deprecated
    @Override
    public BaseBounds impl_getBounds(BaseBounds bounds,
                                     BaseTransform tx,
                                     Node node,
                                     BoundsAccessor boundsAccessor) {
        return getInputBounds(bounds, tx, node, boundsAccessor, getInput());
    }

    /**
     * @treatAsPrivate implementation detail
     * @deprecated This is an internal API that is not intended for use and will be removed in the next version
     */
    @Deprecated
    @Override
    public Effect impl_copy() {
        SepiaTone st = new SepiaTone(this.getLevel());
        st.setInput(this.getInput());
        return st;

    }
}
